# Temperature Control Test Functions
#=====================================================================================    
# High Level Functions to import in GroundControl: predefined Tasks
# 2024-01-26/Kay Struebing
# Last Changes:
# - Integrate low-level-functions and dicts from fle_sys


# Imports
# =============================================================================


import asyncio
import os
import json
import time
from fleming.common.firmware_util import EEFAnalogInput
from fleming.common.firmware_util import get_fan_control_endpoint
from fleming.common.firmware_util import get_node_endpoint

from pylog.pylogger import PyLogger

from py_pli.pylib import VUnits
from py_pli.pylib import send_msg
from py_pli.pylib import GlobalVar

# Global Constants
# =============================================================================

PATH, SCRIPT = os.path.split(__file__)
BASE_NAME = SCRIPT.split('.')[0]


# Hardware Mapping
# =============================================================================

# Fans
# 2021-11-26/kstruebing
# node added
fans = {
    'base1' : {'node': 'fmb_fan','channel': 2}, # FAN 3 
    'base2' : {'node': 'fmb_fan','channel': 3}, # FAN 4 
    'base3' : {'node': 'fmb_fan','channel': 4}, # FAN 5 
#    'uslum' : {'node': 'fmb_fan','channel': 5}, # FAN 6 (not connected yet)
    'int'   : {'node': 'fmb_fan','channel': 0}, # FAN 1
    'ext'   : {'node': 'fmb_fan','channel': 1}, # FAN 2
    'pc'    : {'node': 'fmb_fan','channel': 6}, # FAN 7 (FMB Rear side)
    'fl'    : {'node': 'eef_fan','channel': 0}, # FAN 0 (connected to EEF)
    'trf'   : {'node': 'fmb_fan','channel': 7}, # FAN SW
}

# Temperature Sensors
# 2024-01-26/kstruebing
# fmb was Mainboard, eef was EEFNode

temp_sensors = { # tmb = temp.-measurement-board
    'tmb_upper' : {'node': 'fmb', 'number': 0, 'conv': 256}, # Measurement chamber ceiling
    'tmb_lower' : {'node': 'fmb', 'number': 1, 'conv': 256}, # Measurement chamber bottom
    'tmb_in'    : {'node': 'fmb', 'number': 2, 'conv': 256}, # Airventilation unit inlet to internal fan
    'tmb_out'   : {'node': 'fmb', 'number': 3, 'conv': 256}, # Airventilation unit outlet
#    'tmb_rt'    : {'node': 'fmb', 'number': 4}, # Room Temperature at Instrument inlet (bottom)
    'tmb_fl'    : {'node': 'fmb', 'number': 5, 'conv': 256}, # Flashlamp Fan
    'tmb_int'   : {'node': 'fmb', 'number': 6, 'conv': 256}, # AirV. Peltier Cold side
    'tmb_ext'   : {'node': 'fmb', 'number': 7, 'conv': 256}, # AirV. Peltier Hot side
    'tmb_am'    : {'node': 'fmb', 'number': 16, 'conv': 175}, # Ambient Temperature Sensor
    'tmb_hum'   : {'node': 'fmb', 'number': 17, 'conv': 100}, # Humidity Sensor
#    'tmb_p1'    : {'node': 'eef 'number': 9, 'conv': 256},   # PMT 1 Temp, EEF Rev. 3
    'tmb_p1'    : {'node': 'eef', 'number': 15, 'conv': 256},   # PMT 1 Temp, EEF Rev. 4
#    'tmb_p2'    : {'node': 'eef', 'number': 10, 'conv': 256},  # PMT 2 Temp, EEF Rev. 3
    'tmb_p2'    : {'node': 'eef', 'number': 16, 'conv': 256},  # PMT 2 Temp, EEF Rev. 4
}

# Temperature-Elements
# FLE-1920 2022-07-06
temp_elements = {
    'htu'   : 0, # Upper heating
    'htl'   : 1, # Lower heating
    'htb'   : 2, # Dispenser bottle heating
    'tec'   : 3, # 
#    'av2_2' : 4, # AVU Peltier 2, Pin 2
#    'av2_1' : 5, # AVU Peltier 2, Pin 1
#    'av1_2' : 6, # AVU Peltier 1, Pin 2
#    'av1_1' : 7, # AVU Peltier 1, Pin 1
    'av1'   : 15, # AVU Peltier Pin 1
    'av2'   : 16, # AVU Peltier Pin 2
    'pmt_1' : 9, # PMT 1 Peltier
    'pmt_2' : 10, # PMT 1 Peltier
}

pwm_status = {
    'AV_otw'   : 11, # PWM 2 Driver (AV1+ 2) Overtemperature Warning active low
    'AV_fault' : 10, # PWM 2 Driver (AV1+ 2) Fault active low
    'HT_otw'   : 9, # PWM 1 Driver (HTL+ U) Overtemperature Warning active low
    'HT_fault' : 8, # PWM 1 Driver (HTL+ U) Fault active low
    'PMT_AL_otw'  : 25, # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC overtemperature warning (active low)
    'PMT_AL_fault': 24 # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC fault signal (active low)
}

# Low Level Functions
# =============================================================================

## Logger and Output Text
# -----------------------------------------------------------------------------

async def talk(test_name, msg):
    # msg to Logger and GC Output
    message = f'{test_name}: {msg}'
    PyLogger.logger.info(message)
    await send_msg(json.dumps({'result': message}))

async def gc_out(test_name, msg):
    # msg to GC Output Window
    message = f'{test_name}: {msg}'
    await send_msg(json.dumps({'result': message}))

# Sensor requests
# -----------------------------------------------------------------------------

async def get_tmp(tmb_name):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...
    """
    TMB: Readout of single TMB-Sensor
    tmb_name:
    tmb_upper: Upper Heating
    tmb_lower: Lower Heating
    tmb_in:    AVU internal Fan inlet 
    tmb_out:   AVU internal Fan outlet
    tmb_fl:    Flashlamp Temp
    tmb_int:   AVU Internal Fan (Cool side Peltier) 
    tmb_ext:   AVU external Fan (Hot side Peltier)
    tmb_am:    Ambient Temp
    tmb_hum:   Humidity Sensor
    tmb_p1:    PMT 1  
    tmb_p2:    PMT 2
    """
    if tmb_name not in temp_sensors:
        raise ValueError(f"{tmb_name} not valid.")
    
    # node = VUnits.instance.hal.nodes[temp_sensors[tmb_name]['node']]
    node = get_node_endpoint(temp_sensors[tmb_name]['node'])
    value = (await node.GetAnalogInput(temp_sensors[tmb_name]['number']))[0] * int(temp_sensors[tmb_name]['conv'])
    return value

async def get_all_tmp():
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    """
    Returns all temp-values in one string
    
    TMB: Readout of all TMB-Sensor
    tmb_name:
    tmb_upper: Upper Heating
    tmb_lower: Lower Heating
    tmb_in:    AVU internal Fan inlet 
    tmb_out:   AVU internal Fan outlet
    tmb_rt:    Ambient Temp
    tmb_fl:    Flashlamp Temp
    tmb_int:   AVU Internal Fan (Cool side Peltier)
    tmb_am:    Ambient Temp
    tmb_hum:   Humidity Sensor
    tmb_ext:   AVU external Fan (Hot side Peltier)
    tmb_p1:    PMT 1  
    tmb_p2:    PMT 2
    """
    result = {}
    for tmb in temp_sensors:
        value = await get_tmp(tmb)
        result[tmb] = float(f'{value:.2f}')
        message = f'{tmb}     : {value:.2f}'
        PyLogger.logger.info(message)
        await send_msg(json.dumps({'result': message}))
    return result


async def get_pwm_status(signal = 'AV_fault', board = 'fmb'):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...
    """
    signal: Readout of PWM Status
    'AV_otw'      : 11, # PWM 2 Driver (AV1+ 2) Overtemperature Warning
    'AV_fault'    : 10, # PWM 2 Driver (AV1+ 2) Fault
    'HT_otw'      : 9, # PWM 1 Driver (HTL+ U) Overtemperature Warning
    'HT_fault'    : 8, # PWM 1 Driver (HTL+ U) Fault
    'PMT_AL_otw'  : 25, # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC overtemperature warning (active low)
    'PMT_AL_fault': 24 # PWM driver PMT1-TEC, PMT2-TEC and Alpha Laser TEC fault signal (active low)
    
    """
    if signal not in pwm_status:
        raise ValueError(f"{signal} not valid.")
    
    control = get_node_endpoint(board)
    value = (await control.GetDigitalInput(pwm_status[signal]))[0]
    return value

# Actor control
# -----------------------------------------------------------------------------

async def tec_on(pwm = 65, fan = 10):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    min_pwm = 0.05
    # fmb = VUnits.instance.hal.nodes['fmb']
    fmb = get_node_endpoint('fmb')
    await fan_pwm('int', fan)
    await fan_pwm('ext', 100)
#    p_11 = temp_elements['av1_1']
#    p_12 = temp_elements['av1_2']
#    p_21 = temp_elements['av2_1']
#    p_22 = temp_elements['av2_2']
    p_1 = temp_elements['av1']
    p_2 = temp_elements['av2']

#    await fmb.SetAnalogOutput(p_11, 0.0)
#    await fmb.SetAnalogOutput(p_12, (pwm / 100))
#    await fmb.SetAnalogOutput(p_21, 0.0)
#    await fmb.SetAnalogOutput(p_22, (pwm / 100))
    await fmb.SetAnalogOutput(p_1, (min_pwm))
    await fmb.SetAnalogOutput(p_2, (pwm / 100) + min_pwm)

    return('AVU TE switched on')

async def tec_off():
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # fmb = VUnits.instance.hal.nodes['fmb']
    fmb = get_node_endpoint('fmb')
    min_pwm = 0.05
#   Hardware Change on FMB:
#    p_11 = temp_elements['av1_1']
#    p_12 = temp_elements['av1_2']
#    p_21 = temp_elements['av2_1']
#    p_22 = temp_elements['av2_2']    
    p_1 = temp_elements['av1']
    p_2 = temp_elements['av2']

#    await fmb.SetAnalogOutput(p_11, 0.0)
#    await fmb.SetAnalogOutput(p_12, 0.0)
#    await fmb.SetAnalogOutput(p_21, 0.0)
#    await fmb.SetAnalogOutput(p_22, 0.0)
    await fmb.SetAnalogOutput(p_1, min_pwm)
    await fmb.SetAnalogOutput(p_2, min_pwm)

    return('AVU TEC switched off.')
    
async def h_on(lower_pwm = 96, upper_pwm = 96, fan = 10):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # fmb = VUnits.instance.hal.nodes['fmb']
    fmb = get_node_endpoint('fmb')
    h_lower = temp_elements['htl']
    h_upper = temp_elements['htu']
    await fan_pwm('int', fan)
    await fmb.SetAnalogOutput(h_lower, lower_pwm / 100.0)
    await fmb.SetAnalogOutput(h_upper, upper_pwm / 100.0)
    return('Lower, Upper Heating switched ON.')

async def h_off():
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # Turns Heating foils OFF
    # [] tested          

    # fmb = VUnits.instance.hal.nodes['fmb']
    fmb = get_node_endpoint('fmb')
    h_lower = temp_elements['htl']
    h_upper = temp_elements['htu']
    await fmb.SetAnalogOutput(h_lower, 0.0)
    await fmb.SetAnalogOutput(h_upper, 0.0)
    return('Lower, Upper Heating switched OFF.')

async def p_on(ch = 1, pwm = 30):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # eef = VUnits.instance.hal.nodes['EEFNode']
    eef = get_node_endpoint('eef')
    if ch == 1:
        await eef.SetAnalogOutput(temp_elements['pmt_1'], (pwm / 100))
    else:
        await eef.SetAnalogOutput(temp_elements['pmt_2'], (pwm / 100))
    
async def p_off(ch = 1):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # eef = VUnits.instance.hal.nodes['EEFNode']
    eef = get_node_endpoint('eef')
    if ch == 1:
        await eef.SetAnalogOutput(temp_elements['pmt_1'], 0.0)
    else:
        await eef.SetAnalogOutput(temp_elements['pmt_2'], 0.0)

# EMERGENCY OFF
async def e_off():
    await h_off()
    await tec_off()
    return('All Temperature Elements switched OFF.')

# Fan Control
async def fan_pwm(fan_name, pwr):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [ ] GC Test 
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    """
    fan_name: base1, base2, base3, int, ext, pc, fl, trf
    pwr: 0...100%
    """
    if (pwr < 0) or (pwr > 100):
        raise ValueError(f"pwr must be in the range [0, 100]")

    fan = get_fan_control_endpoint(fans[fan_name]['node'])
    channel = fans[fan_name]['channel']
    await fan.SetSpeed(channel, pwr, timeout=1)
    await fan.Enable(channel, (pwr > 0), timeout=1)
    return f"Fan '{fan_name}' @ {int(pwr)}%"


# High level module tests 
# =============================================================================

async def tec_test(power = 60, limit = -2.0):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [x] GC Test 2024-01-26 (Newport)
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # starts TEC for 60 secs and detects temperature gradient
    # ============ GC Ready =================================
    
    TEST_NAME = 'tec_test'
    DURATION = 60
    SAMPLE_RATE = 1
    FAN = 10
    T_MAX = 30
    limit_f = float(limit) if limit else -2.0

    bar = lambda x, x_max, width = 80, char = "|" : char.ljust(int(x / x_max * width), char)
    t_cool_1 = await get_tmp('tmb_int')
    t_hot_1 = await get_tmp('tmb_ext')
    msg = await tec_on(power, FAN)
    await talk(TEST_NAME, msg)
    start_on = time.time() 
    GlobalVar.set_stop_gc(False)
    while (time.time() - start_on) < DURATION:
        if GlobalVar.get_stop_gc():
            await tec_off()
            msg = "gc_predefined_tasks> Script stopped by user"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))
            return msg
        if await get_pwm_status('AV_fault') == 0:
            await tec_off()
            msg = "ERROR: AVU PWM Overtemperature Shutdown, Script stopped"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))
            return msg
        if await get_pwm_status('AV_otw') == 0:
            await tec_off()
            msg = "WARNING:: AVU PWM Overtemperature, Script continued"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))

        t_cool = await get_tmp('tmb_int')
        print(f'{t_cool:.2f} \t {bar(t_cool, T_MAX)}')
        result=bar(t_cool, T_MAX)
        msg = {"result": f"{t_cool:.2f}deg C \t {result}"}
        await send_msg(json.dumps(msg))        
        await asyncio.sleep(SAMPLE_RATE)
    msg = await tec_off()
    await talk(TEST_NAME, msg)
    d_t_cool = await get_tmp('tmb_int') - t_cool_1
    d_t_hot = await get_tmp('tmb_ext') - t_hot_1
    if d_t_cool > limit_f:
        return f'{TEST_NAME}: TEST FAILED: Result: AVU Cooling Rate: {d_t_cool:.2f}K/min, TEC_Hot: {d_t_hot:.2f}K/min'
    else:
        return f'{TEST_NAME}: TEST OK: Result: AVU Cooling Rate: {d_t_cool:.2f}K/min, TEC_Hot: {d_t_hot:.2f}K/min'

async def htu_test(limit = 2.0):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [x] GC Test 2024-01-26 (Newport)
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # starts Upper heating and detects temperature gradient
    # starts TEC for 60 secs and detects temperature gradient
    # ============ GC Ready =================================

    TEST_NAME = 'htu_test'
    DURATION = 60
    SAMPLE_RATE = 1
    POWER = 96
    FAN = 10
    T_MAX = 70
    limit_f = float(limit) if limit else 2.0

    bar = lambda x, x_max, width = 80, char = "|" : char.ljust(int(x / x_max * width), char)
    t_upper_1 = await get_tmp('tmb_upper')
    msg = await h_on(0, POWER, FAN)
    await talk(TEST_NAME, msg)
    start_on = time.time() 
    GlobalVar.set_stop_gc(False)
    while (time.time() - start_on) < DURATION:
        if GlobalVar.get_stop_gc():
            await h_off()
            PyLogger.logger.error("gc_predefined_tasks> Script stopped by user")
            await send_msg(json.dumps({"result": f"Script stopped by user"}))
            return "Stopped by user"
        if await get_pwm_status('HT_fault') == 0:
            await h_off()
            msg = "ERROR: HTU/HTL PWM Overtemperature Shutdown, Script stopped"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))
            return msg
        if await get_pwm_status('HT_otw') == 0:
            await h_off()
            msg = "WARNING:: HTU/HTL PWM Overtemperature, Script continued"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))

        t_upper = await get_tmp('tmb_upper')
        print(f'{t_upper:.2f} \t {bar(t_upper, T_MAX)}')
        result=bar(t_upper, T_MAX)
        msg = {"result": f"{t_upper:.2f} \t {result}"}
        await send_msg(json.dumps(msg))        
        await asyncio.sleep(SAMPLE_RATE)
    msg = await h_off()
    await talk(TEST_NAME, msg)
    d_t_upper = await get_tmp('tmb_upper') - t_upper_1
    if d_t_upper < limit_f:
        return f'{TEST_NAME}: FAILED, Result: Upper Heating Rate: {d_t_upper:.2f}K/min' 
    else:
        return f'{TEST_NAME}: OK, Result: Upper Heating Rate: {d_t_upper:.2f}K/min'    

async def htl_test(limit = 2.0):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [x] GC Test 2024-01-26 (Newport)
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # starts Upper heating and detects temperature gradient
    # starts TEC for 60 secs and detects temperature gradient
    # 2022-03-18/kstruebing
    # 2022-11-18/kstruebing: Overtemperature error handling
    # ============ GC Ready =================================

    TEST_NAME = 'htl_test'
    DURATION = 60
    SAMPLE_RATE = 1
    POWER = 96
    FAN = 10
    T_MAX = 70
    limit_f = float(limit) if limit else 2.0
    bar = lambda x, x_max, width = 80, char = "|" : char.ljust(int(x / x_max * width), char)
    t_lower_1 = await get_tmp('tmb_lower')
    msg = await h_on(POWER, 0, FAN)
    await talk(TEST_NAME, msg)
    start_on = time.time() 
    GlobalVar.set_stop_gc(False)
    while (time.time() - start_on) < DURATION:
        if GlobalVar.get_stop_gc():
            await h_off()
            PyLogger.logger.error("gc_predefined_tasks> Script stopped by user")
            await send_msg(json.dumps({"result": f"Script stopped by user"}))
            return "Stopped by user"
        if await get_pwm_status('HT_fault') == 0:
            await h_off()
            msg = "ERROR: HTU/HTL PWM Overtemperature Shutdown, Script stopped"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))
            return msg
        if await get_pwm_status('HT_otw') == 0:
            await h_off()
            msg = "WARNING:: HTU/HTL PWM Overtemperature, Script continued"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))

        t_lower = await get_tmp('tmb_lower')
        print(f'{t_lower:.2f} \t {bar(t_lower, T_MAX)}')
        result=bar(t_lower, T_MAX)
        msg = {"result": f"{t_lower:.2f} \t {result}"}
        await send_msg(json.dumps(msg))        
        await asyncio.sleep(SAMPLE_RATE)
    msg = await h_off()
    await talk(TEST_NAME, msg)
    d_t_lower= await get_tmp('tmb_lower') - t_lower_1
    if d_t_lower < limit_f:
        return f'{TEST_NAME}: FAILED, Result: Lower Heating: {d_t_lower:.2f}K/min' 
    else:
        return f'{TEST_NAME}: OK, Result: Lower Heating: {d_t_lower:.2f}K/min'    

async def pmt_tec_test(channel = 1, duration_s = 60, limit = -2.0):
    # [x] Terminal Test 2024-01-26 (Newport)
    # [x] GC Test 2024-01-26 (Newport)
    # Last modified: 2024-01-26/kstruebing
    # get_node_endpoint...

    # starts TEC for duration secs and detects temperature gradient
    # 2022-02-22/kstruebing
    # 2022-03-18/kstruebing
    # ============ GC Ready =================================

    TEST_NAME = 'pmt_tec_test'
    SAMPLE_RATE = 1 #Data output rate
    POWER = 30
    T_MAX = 50


    # Ground Control Conversion Default Values 
    ch = int(channel) if channel else 1
    duration = int(duration_s) if duration_s else 60
    limit_f = float(limit) if limit else -2.0

    if ch == 1:
        tmb = 'tmb_p1'
    else:
        tmb = 'tmb_p2'

    bar = lambda x, x_max, width = 80, char = "|" : char.ljust(int(x / x_max * width), char)
    t_cool_1 = await get_tmp(tmb)
    msg = await p_on(ch, POWER)
    limit_f = float(limit) if limit else -2.0
    await talk(TEST_NAME, msg)
    start_on = time.time() 
    GlobalVar.set_stop_gc(False)
    while (time.time() - start_on) < duration:
        if GlobalVar.get_stop_gc():
            msg = await p_off(ch)
            await send_msg(json.dumps({"result": msg}))
            PyLogger.logger.error("gc_predefined_tasks> Script stopped by user")
            await send_msg(json.dumps({"result": f"Script stopped by user"}))
            return "Stopped by user"
        if await get_pwm_status('PMT_AL_fault', 'eef') == 0:
            await h_off()
            msg = "ERROR: HTU/HTL PWM Overtemperature Shutdown, Script stopped"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))
            return msg
        if await get_pwm_status('PMT_AL_otw', 'eef') == 0:
            await h_off()
            msg = "WARNING:: HTU/HTL PWM Overtemperature, Script continued"
            PyLogger.logger.error(msg)
            await send_msg(json.dumps({"result": msg}))

        t_cool = await get_tmp(tmb)
        print(f'{t_cool:.2f} \t {bar(t_cool, T_MAX)}')
        result=bar(t_cool, T_MAX)
        msg = {"result": f"{t_cool:.2f}deg C \t {result}"}
        await send_msg(json.dumps(msg))        
        await asyncio.sleep(SAMPLE_RATE)
    msg = await p_off(ch)
    await talk(TEST_NAME, msg)
    d_t_cool = (await get_tmp(tmb) - t_cool_1) / duration * 60
    if d_t_cool > limit_f:
        return f'{TEST_NAME}: FAILED, Result: PMT Channel {ch} Cooling Rate: {d_t_cool:.2f}K/min' 
    else:
        return f'{TEST_NAME}: OK, Result: PMT channel {ch} Cooling Rate: {d_t_cool:.2f}K/min'    
